﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.Common.Extensions;
using VA.PPMS.IWS.MappingService.Helpers;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public class MapDeaScheduleToCrm : MapperRelatedBase
    {
        public MapDeaScheduleToCrm(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
            : base(ppmsContextHelper, ppmsHelper)
        {}

        public async Task<ppms_deascheduleprivilege> MapInsert(DeaSchedulePrivilege schemaEntity, Account provider)
        {
            ValidateEntity(schemaEntity);

            // map entity
            var entity = new ppms_deascheduleprivilege
            {
                Id = Guid.NewGuid(),
                ppms_deanumber = schemaEntity.DeaNumber,
                ppms_verifier = schemaEntity.Verifier
            };

            if (schemaEntity.ExpirationDateSpecified && schemaEntity.ExpirationDate.Year > EnumHelper.MinYear)
                entity.ppms_expirationdate = schemaEntity.ExpirationDate.ToCrmDate();

            if (schemaEntity.VerificationDateSpecified && schemaEntity.VerificationDate.Year > EnumHelper.MinYear)
                entity.ppms_verificationdate = schemaEntity.VerificationDate.ToCrmDate();

            if (schemaEntity.HasScheduleIiSpecified)
                entity.ppms_hasscheduleii = schemaEntity.HasScheduleIi;

            if (schemaEntity.HasScheduleIiNonNarcotic)
                entity.ppms_hasscheduleiinonnarcotic = schemaEntity.HasScheduleIiNonNarcotic;

            if (schemaEntity.HasScheduleIiiSpecified)
                entity.ppms_hasscheduleiii = schemaEntity.HasScheduleIii;

            if (schemaEntity.HasScheduleIiiNonNarcoticSpecified)
                entity.ppms_hasscheduleiiinonnarcotic = schemaEntity.HasScheduleIiiNonNarcotic;

            if (schemaEntity.HasScheduleIvSpecified)
                entity.ppms_hasscheduleiv = schemaEntity.HasScheduleIv;

            if (schemaEntity.HasScheduleVSpecified)
                entity.ppms_hasschedulev = schemaEntity.HasScheduleV;

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) entity.OwnerId = Owner;

            await Task.Run(() => {});

            return entity;
        }

        public async Task<ppms_deascheduleprivilege> MapUpdate(DeaSchedulePrivilege entity, Account provider)
        {
            // get entity
            var crmEntity = GetCrmEntity(provider, entity.DeaNumber);

            // matching account not found
            if (crmEntity == null)
            {
                return await MapInsert(entity, provider);
            }

            ValidateEntity(entity);

            // map fields
            var newEntity = new ppms_deascheduleprivilege()
            {
                Id = crmEntity.Id,
                ppms_deanumber = crmEntity.ppms_deanumber,
                ppms_expirationdate = crmEntity.ppms_expirationdate,
                ppms_verificationdate = crmEntity.ppms_verificationdate,
                ppms_hasscheduleii = crmEntity.ppms_hasscheduleii,
                ppms_hasscheduleiinonnarcotic = crmEntity.ppms_hasscheduleiinonnarcotic,
                ppms_hasscheduleiii = crmEntity.ppms_hasscheduleiii,
                ppms_hasscheduleiiinonnarcotic = crmEntity.ppms_hasscheduleiiinonnarcotic,
                ppms_hasscheduleiv = crmEntity.ppms_hasscheduleiv,
                ppms_hasschedulev = crmEntity.ppms_hasschedulev
            };

            if (IsChanged(entity.DeaNumber, crmEntity.ppms_deanumber))
                newEntity.ppms_deanumber = entity.DeaNumber;

            if (entity.ExpirationDateSpecified && entity.ExpirationDate.Year > EnumHelper.MinYear)
                newEntity.ppms_expirationdate = entity.ExpirationDate.ToCrmDate();

            if (entity.VerificationDateSpecified && entity.VerificationDate.Year > EnumHelper.MinYear)
                newEntity.ppms_verificationdate = entity.VerificationDate.ToCrmDate();

            if (entity.HasScheduleIiSpecified)
                newEntity.ppms_hasscheduleii = entity.HasScheduleIi;

            if (entity.HasScheduleIiNonNarcotic)
                newEntity.ppms_hasscheduleiinonnarcotic = entity.HasScheduleIiNonNarcotic;

            if (entity.HasScheduleIiiSpecified)
                newEntity.ppms_hasscheduleiii = entity.HasScheduleIii;

            if (entity.HasScheduleIiiNonNarcoticSpecified)
                newEntity.ppms_hasscheduleiiinonnarcotic = entity.HasScheduleIiiNonNarcotic;

            if (entity.HasScheduleIvSpecified)
                newEntity.ppms_hasscheduleiv = entity.HasScheduleIv;

            if (entity.HasScheduleVSpecified)
                newEntity.ppms_hasschedulev = entity.HasScheduleV;

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) newEntity.OwnerId = Owner;

            return newEntity;
        }

        public SetStateRequest MapDelete(ppms_deascheduleprivilege crmItem)
        {
            if (crmItem != null)
            {
                return DeactivateEntity(crmItem);
            }

            return null;
        }

        private static void ValidateEntity(DeaSchedulePrivilege entity)
        {
            // validate entity
            var messages = ValidationHelper.Validate(entity);
            if (messages.Any())
            {
                // gather error messages
                var sb = new StringBuilder();
                foreach (var item in messages) sb.Append($"{item}; ");

                throw new PpmsServiceException($"DEA number exception: {sb.ToString()}");
            }
        }

        private static ppms_deascheduleprivilege GetCrmEntity(Account provider, string id)
        {
            ppms_deascheduleprivilege found = null;
            var list = provider.ppms_account_deascheduleprivilege;
            if (list != null)
            {
                found = list.FirstOrDefault(x => x.ppms_deanumber == id);
            }

            return found;
        }

        private static DeaSchedulePrivilege ConvertEntity<T>(T entity)
        {
            return (DeaSchedulePrivilege)Convert.ChangeType(entity, typeof(DeaSchedulePrivilege));
        }

        public override async Task<Entity> MapUpdate<T>(T entity, Entity parent)
        {
            return await MapUpdate(ConvertEntity(entity), (Account)parent);
        }

        public override async Task<Entity> MapInsert<T>(T entity, Entity parent)
        {
            return await MapInsert(ConvertEntity(entity), (Account)parent);
        }

        public override void AddChildrenToProvider(IList<Entity> entities, Entity parent)
        {
            if (IsWithinContext) AssociateRelatedEntities(parent, entities, "ppms_account_deascheduleprivilege");
            else
            {
                if (entities != null && entities.Count > 0)
                {
                    var account = (Account)parent;
                    if (account != null) account.ppms_account_deascheduleprivilege = ConvertEntityList<ppms_deascheduleprivilege>(entities);
                }
            }
        }

        public override IEnumerable<SetStateRequest> MapDelete<T>(IList<T> entities, Entity parent)
        {
            if (entities == null || !entities.Any()) return null;

            // Check provider
            var provider = (Account)parent;
            if (provider == null) return null;

            var list = new List<ppms_deascheduleprivilege>();
            var deas = provider.ppms_account_deascheduleprivilege.ToList();
            DeaSchedulePrivilege entity;

            if (deas != null && deas.Any())
            {
                // Map schema entities for delete
                foreach (var item in entities)
                {
                    entity = ConvertEntity(item);
                    var matches = provider.ppms_account_deascheduleprivilege.Where(p => p.ppms_deanumber == entity.DeaNumber);
                    list.AddRange(matches);
                }
            }

            return EntityDelete((IEnumerable<ppms_deascheduleprivilege>)list);
        }
    }
}